/*******************************************************************************
This file is part of WhiteStork.

WhiteStork is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

WhiteStork is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with WhiteStork; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006 ComArch S.A.
*******************************************************************************/
#ifndef   _DICTIONARY_ENGINE_BASE
#define   _DICTIONARY_ENGINE_BASE

#ifdef    __cplusplus
       extern "C" {
#endif


//______________________________________________________________________________
// *****************************************************************************
//************************************************************* HEADERS SECTION:
//------------------------------------------------------------------------------
// header with gnu library for C - most of types and functions needed it (all
// started from g* and g_*().
#include <glib.h>
//------------------------------------------------------------------------------

#define _GLOBAL_FUNCTIONS_NAME_ "engine_global_functions"

//______________________________________________________________________________
// *****************************************************************************
//******************************************************* GLOBAL MACROS SECTION:
//------------------------------------------------------------------------------
/** \brief Checks if module is peaceable with concrete location.
 *
 * If user of engine module want to know if concrete module can handle concrete 
 * location, he can call <i>dict_eng_module_check()</i>and check the result 
 * of this function.
 *
 * @param module :: <b>EngineModule</b> - module engine which will be testing
 * @param location :: <b>gchar*</b> - location on wich engine will be testing
 * @return <i><b>gboolean</b></i> :: tells if location <i>location</i> can be 
 * handled by module <i>module</i>
 */
#define         dict_eng_module_check(module,location)   \
                        ( (module).engine_check( (location) ) )
//------------------------------------------------------------------------------
/** \brief Get short description of engine module.
 *
 * If there is need to present to the user of end programm some information 
 * about enigne he is using, we can call dict_eng_module_get_description() and
 * print/show the result of this call.
 *
 * @param module :: <b>EngineModule</b>- module engine which description we want
 * @return <i><b>gchar*</b></i> :: there is string, decribing module engine 
 * <i>module</i>, in this pointer. User should NOT free this pointer as it 
 * points to static table of chars
 */
#define         dict_eng_module_get_description(module)   \
                        ( (module).engine_description() )
//------------------------------------------------------------------------------
/** \brief Get version of engine module.
 *
 * Sometimes it could be useful to get know what version of engine module we are
 *  using. We can call this function to retrieve these informations from module.
 * Note that version information could be also placed in description of module, 
 * but here user will get only string representing version, what could be needed
 * in some situation (e.g.: if end-user will have two the same module, then 
 * programm has to decide which should be used, or ask user).
 *
 * @param module :: <b>EngineModule</b> - module engine which implementation 
 * version we want
 * @return <i><b>gchar*</b></i> :: pointer to string representing module 
 * <i>module</i>version. User should NOT free this pointer as it points to 
 * static table of chars
 */
#define         dict_eng_module_get_version(module)   \
                        ( (module).engine_version() )
//------------------------------------------------------------------------------
/** \brief Get format on which working engnine module.
 *
 * If we want to know on what dictionary format works current engine module, we 
 * can call this function. As a result we will retrieve string telling us what 
 * format-based is concrete engine module (e.g. XDXF, WordNet etc.).
 *
 * @param module :: <b>EngineModule</b> - engine module which dictionary format
 * we want to get
 * @return <i><b>gchar*</b></i> :: string representing format of dictionary on 
 * which module <i>module</i> can works. User should NOT free this pointer as 
 * it points to static table of chars
 */
#define         dict_eng_module_get_format(module)   \
                        ( (module).engine_version() )
//------------------------------------------------------------------------------
/** \brief Open concrete dictionary.
 *
 * This functions opens location (this could mean: filename, URL, any other way 
 * to specify concrete localization of dictionary database). To be more sure 
 * that this function will be sucefull, user can call before 
 * dict_eng_module_check() to check if this module can handle dictionary under 
 * passed location. But this do not warrant that calling 
 * dict_eng_module_create() will not fail. User should check the result of 
 * function to be sure that everything was ok.
 *
 * @param module :: <b>EngineModule</b> - engine module which should be used to 
 * open location <i>location</i>
 * @param location :: <b>gchar*</b> - location of dictionary database (file, 
 * directory, URL etc.)
 * @param flags :: <b>EngineOptimizationFlag</b> - flags defines how program 
 * shoud automaticlly use optimization methods for opened dictionary
 * @return <i><b>Engine*</b></i> :: pointer to new enigne structure; After use, 
 * if this structure will not be needed anymore, user should call 
 * dict_eng_destroy() to free memory
 * \sa EngineOptimizationFlag :|: dict_eng_module_check() :|: EngineModule :|: 
 * Engine
*/
#define         dict_eng_module_create(module,location,flags)   \
        ( (module).engine_create(  (location), (flags), NULL, NULL, 0.01 ) )
//------------------------------------------------------------------------------
/** \brief Open concrete dictionary - define handler for signal of progress
 *
 * This functions is similiar to dict_eng_module_create(module,location,flags).
 * In addition to it, this function get additional parameters, defining how
 * engine should "inform" the user about progress of eventuall caching proccess.
 *
 * @param module :: <b>EngineModule</b> - engine module which should be used to 
 * open location <i>location</i>
 * @param location :: <b>gchar*</b> - location of dictionary database (file, 
 * directory, URL etc.)
 * @param flags :: <b>EngineOptimizationFlag</b> - flags defines how program 
 * shoud automaticlly use optimization methods for opened dictionary
 * @param progress_handler :: <b>cb_progress</b> - handler which will be called 
 * from time to time, to give to the user any response about caching progress
 * @param progress_data :: <b>gpointer</b> - pointer to data which will be given
 * as a user_data argument for progress_data function during its calling
 * @param seed :: <b>gdouble</b> - defines after how big step (in percentage)
 * in progress of caching, engine send new signal - call progress_handler with
 * actual value of progress
 * @return <i><b>Engine*</b></i> :: pointer to new enigne structure; After use, 
 * if this structure will not be needed anymore, user should call 
 * dict_eng_destroy() to free memory
 * \sa EngineOptimizationFlag :|: dict_eng_module_check() :|: EngineModule :|: 
 * Engine :|: cb_progress
 */
#define         dict_eng_module_create_ext(module,                      \
                                           location,                    \
                                           flags,                       \
                                           progress_handler,            \
                                           progress_data,               \
                                           seed                         \
                                           )  (                         \
                       (module).engine_create( (location),              \
                                               (flags),                 \
                                               (progress_handler),      \
                                               (progress_data),         \
                                               (seed)                   \
                                             ))
//------------------------------------------------------------------------------


//______________________________________________________________________________
// *****************************************************************************
//****************************************** CONCRETE DICTIONARY MACROS SECTION:
//------------------------------------------------------------------------------
#define         dict_eng_set_progress_seed(engine, signal, val) (              \
                        (engine) -> engine_set_progress_seed( (engine),        \
                                                              (signal),        \
                                                              (val)            \
                                                            ))
//------------------------------------------------------------------------------

#define         dict_eng_set_auto_free(engine, state) ( \
                        (engine) -> engine_set_auto_free( (engine), (state) )
//------------------------------------------------------------------------------
/** \brief Activate optimization mechanizms for concrete engine.
 *
 * Every engine could have some optimization methods. By calling this function 
 * we enabled this option. For the user of engine it is not important what kind
 * of optimization is used in current engine, how does it work etc. It should 
 * be opaque for engine's users.
 *  
 * @param engine :: <b>Engine*</b> - pointer to engine which should optimized
 * @return <i><b>void</b></i> :: nothing. This function only take a chance to 
 * make engine working faster, but it is not said that every step 
 * will be succesful. But even if not - engine has to work properly.
 * \sa dict_eng_module_create() :|: dict_eng_is_optimized() :|: Engine
 */
#define         dict_eng_optimize(engine)  \
                        ((engine) -> engine_optimize( (engine) ))
//------------------------------------------------------------------------------
/** \brief Checks if current engine has activated optimization.
 *
 * User can call this function to get know if concrete engine has enabled 
 * optimization. Optimization can be activated by calling dict_eng_optimize() 
 * or passing ENGINE_CREATE or ENGINE_REFRESH flags to dict_eng_module_create()
 * while opening concrete dictionary. 
 *  
 * @param engine :: <b>Engine*</b> - pointer to engine which should be checked 
 * if is optimized now
 * @return <i><b>gboolean</b></i> :: nothing. This function only take a chance 
 * to make engine working faster, but it is not said that every step 
 * will be succesful
 * \sa dict_eng_optimize() :|: Engine :|: EngineOptimizationFlag
 */
#define         dict_eng_is_optimized( engine )  \
                        ((engine) -> engine_is_optimized( (engine) ))
//------------------------------------------------------------------------------
/** \brief To get know from what location was created current dictionary.
 *
 * If user of engine want to get know from what locatione exactly was created 
 * concrete engine, he could call this function. It returns string, most likely 
 * the same as the one passed earlier to dict_eng_module_create() function 
 * as a <i>location</i> argument.
 *  
 * @param engine :: <b>Engine*</b> - dictionary from which we want to get 
 * location 
 * @return <i><b>gchar*</b></i> :: pointer to chars, from which concrete 
 * dictionary was created
 * \sa dict_eng_module_create()
 */
#define         dict_eng_get_location(engine)    \
                        ((engine) -> engine_location( (engine) ))
//------------------------------------------------------------------------------
/** \brief Tells if last operation was sucesfull.
 *
 * If user want to be sure that everything was ok, during last operation 
 * (it has finished with success), he has to check if dict_eng_get_last_status()
 * is equal to ENGINE_NO_ERROR. If not, he can also find the reason 
 * why operation failed.
 * @param engine :: <b>Engine*</b> - dictionary which last operation's returned 
 * code we want to get
 * @return <i><b>EngineStatus</b></i> :: finish code of last operation
 * \sa EngineStatus :|: dict_eng_state_message()
 */
#define         dict_eng_get_last_status(engine)    \
                        ((engine) -> engine_error( (engine) ))
//------------------------------------------------------------------------------
/** \brief Translate pure state code to meaningfull message.
 *
 * If there was a error during engine was working, we can present to the user 
 * the reason for this error in meaningfull form.
 *
 * @param error :: <b>EngineState</b> - code which we want to translate
 * @return <i><b>gchar*</b></i> :: message describing concrete error code
 * \sa EngineStatus :|: dict_eng_state_message()
 */
#define         dict_eng_state_message(error)     \
                        ((engine) -> engine_error_message( (error) ))
//------------------------------------------------------------------------------
/** \brief Destroy gently and completly current dictionary.
 *
 * When user do not need anymore concrete dictionary, he must destroy it to 
 * free memory allocated by this dictionary. It will close avery opened file 
 * descriptor, free any additional memory block allocated while engine was 
 * working etc. User are not allowed to call free() on dictionary without 
 * calling before dict_eng_destroy().
 *  
 * @param engine :: <b>Engine*</b> - pointer to dictionary which is not
 * needed anymore
 * @return <i><b>void</b></i> :: nothing. This function should work properly,
 * and free any resources which arelady are in use by current dictionary.
 * \sa dict_eng_module_create() :|: Engine
 */
#define         dict_eng_destroy(engine)    \
                        ((engine) -> engine_close( (engine) ))
//------------------------------------------------------------------------------
/** \brief Start searching for words list.
 *
 * This function starts searching for words list. User should set before 
 * callback for ENGINE_WORD_LIST_SIGNAL, to be able to retrieve words list, 
 * because this function do not return any value.
 *  
 * @param engine :: <b>Engine*</b> - pointer to dictionary in which we 
 * are looking for words matching to pattern
 * @param pattern :: <b>gchar*</b> - string representing pattern to which 
 * returned word should be matching
 * @return <i><b>void</b></i> :: nothing. Result of searching can be retrieved 
 * by callback for signal ENGINE_WORD_LIST_SIGNAL
 * \sa dict_eng_search_word_translation() :|: 
 * dict_eng_search_word_translation_extended() :|: dict_eng_set_callback() :|: 
 * ENGINE_WORD_LIST_SIGNAL
 */
#define         dict_eng_search_word_list(engine,pattern)    \
                        ((engine) -> engine_search_word_list( (engine),        \
                                                              (pattern)        \
                                                            ))
//------------------------------------------------------------------------------
/** \brief Start searching for word translation in dictionary.
 *
 * Start searching for translation of word. To be able to retrieve translation 
 * user should use callback for ENGINE_WORD_TRANSLATION_SIGNAL signal.
 *  
 * @param engine :: <b>Engine*</b> - dictionary in which we want find word's 
 * translation
 * @param word :: <b>gchar*</b> - string representing word which translation 
 * we are looking for
 * @return <i><b>void</b></i> :: nothing. Result of searching can be retrieved 
 * by callback for signal ENGINE_WORD_TRANSLATION_SIGNAL
 * \sa dict_eng_search_word_list() :|: 
 * dict_eng_search_word_translation_extended() :|: dict_eng_set_callback() :|: 
 * ENGINE_WORD_TRANSLATION_SIGNAL
 */
#define         dict_eng_search_word_translation(engine,word)    \
                        ((engine) -> engine_search_word_translation( (engine), \
                                                                     (word)    \
                                                                   ))
//------------------------------------------------------------------------------
/** \brief Start searching for extended word translation in dictionary.
 *
 * Start searching for extended  translation of word. To be able to retrieve 
 * translation user should use callback for ENGINE_WORD_TRANSLATION_SIGNAL 
 * signal. Extended means that retrieved translation's string could be some more
 * than plain text (e.g. HTML, Image, etc.). User of engine should know what 
 * kind of data will retrieve ofter calling this method.
 *
 * \todo This option are in projecting phase. In the future callback for 
 * ENGINE_WORD_TRANSLATION_SIGNAL should take parameter <i>gpointer</i> than 
 * <i>gchar*</i> to make possible for user of engine to retrieve data different 
 * than gchar*. There shold be also some additional function letting know user 
 * what kind of data he will retrieve.
 *  
 * @param engine :: <b>Engine*</b> - dictionary in which we want find word's 
 * translation
 * @param word :: <b>gchar*</b> - string representing word which translation 
 * we are looking for
 * @return <i><b>void</b></i> :: nothing. Result of searching can be retrieved 
 * by callback for signal ENGINE_WORD_TRANSLATION_SIGNAL
 * \sa dict_eng_search_word_list() :|: dict_eng_search_word_translation() :|: 
 * dict_eng_set_callback() :|: ENGINE_WORD_TRANSLATION_SIGNAL
 */
#define         dict_eng_search_word_translation_extended(engine,word)   \
                ((engine) -> engine_search_word_translation_extended((engine), \
                                                                     (pattern) \
                                                                     ))
//------------------------------------------------------------------------------
/** \brief Setting any callback for current dictionary.
 *
 * As long as functions which search for words list or translation do not 
 * return any value, user has to set callback which will be called after 
 * searching end. This make easy calling these functions from other proccess, 
 * threads etc. This function sets callbacks for every signal.
 *  
 * @param engine :: <b>Engine*</b> - dictionary for which we want set some
 * callback
 * @param signal :: <b>gchar*</b> - string representing for what signal we want
 * to set callback
 * @param c_handler :: <b>gpointer</b> - pointer to function which will be 
 * called after signal <i>signal</i> will occured
 * @param data :: <b>gpointer</b> - pointer to user data, which will be always 
 * passed to callback <i>c_handler</i> for signal <i>signal</i>
 * as a last argument
 * @return <i><b>gpointer</b></i> :: pointer to previous callback; if this 
 * callbacks was not set yet, it returns NULL
 * \sa cb_word_list :|: cb_word_translation :|:  ENGINE_WORD_LIST_SIGNAL :|: 
 * ENGINE_WORD_TRANSLATION_SIGNAL
 */
#define         dict_eng_set_callback(engine,signal,c_handler,data)     \
                        ((engine) -> engine_set_callback( (engine),     \
                                                          (signal),     \
                                                          (c_handler),  \
                                                          (data)        \
                                                        ))
//------------------------------------------------------------------------------


//______________________________________________________________________________
// *****************************************************************************
//************************************ SIGNALS DEFINITION FOR CALLBACKS SECTION:
//------------------------------------------------------------------------------
/** \brief Signal definition for callback called when word list was found.
 *
 * ENGINE_WORD_LIST_SIGNAL defines name for signal passed to 
 * dict_eng_set_callback() function as a signal parameter. Function set 
 * to handle this signal should be called from dict_eng_search_word_list() only
 * and have cb_word_list type.
 * <b>NOTE:</b> programmers must not use value of ENGINE_WORD_LIST_SIGNAL 
 * directly!
 * \sa cb_word_list :|: dict_eng_set_callback() :|: dict_eng_search_word_list() 
 * :|: ENGINE_WORD_TRANSLATION_SIGNAL
 */
#define ENGINE_WORD_LIST_SIGNAL         "on_word_list_found"
//------------------------------------------------------------------------------
/** \brief Signal definition for callback called when word trans. was found.
 *
 * ENGINE_WORD_TRANSLATION_SIGNAL defines name for signal passed to 
 * dict_eng_set_callback() function as a signal parameter. Function set 
 * to handle this signal should be called from 
 * dict_eng_search_word_translation() and  
 * dict_eng_search_word_translation_extended() only and have cb_word_translation.
 * <b>NOTE:</b> programmers must not use value of ENGINE_WORD_LIST_SIGNAL 
 * directly!
 * \sa cb_word_translation :|: dict_eng_set_callback() :|: 
 * dict_eng_search_word_translation() :|: ENGINE_WORD_LIST_SIGNAL
 */
#define ENGINE_WORD_TRANSLATION_SIGNAL  "on_word_translation_found"
//------------------------------------------------------------------------------
#define ENGINE_PROGRESS_OPTIMIZING_SIGNAL "on_progress_optimizing"
//------------------------------------------------------------------------------


//______________________________________________________________________________
// *****************************************************************************
//************************************************** DEFINITIONS OF ENUMS TYPES:
//------------------------------------------------------------------------------
/** \brief Codes of flags used to auto optimizing dictionary file.
 * 
 * - Enum type for flags used as a flags parameter for dict_eng_module_create() 
 * function.
 */
typedef enum
{
        /**
         * :: flags says thatoptimization (any methodsof optimalization) should 
         * be enabled. If optimization methods needs to make some files and 
         * these files exists, enigne will just use it (do not refresh).
         */
        ENGINE_CREATE = 0, 
        /**
         * :: flags says that for this particular dictionary programm should 
         * disable optimization.
         */
        ENGINE_NO,
        /**
         * :: the same flags as ENGINE_CREATE, but tells 
         * dict_eng_module_create() function that even if file (or any 
         * resources) for optimizatione methods exists, they should be 
         * recreated/refreshed.
         */
        ENGINE_REFRESH 
} EngineOptimizationFlag;
//------------------------------------------------------------------------------
/** \brief Codes of possible errors which can occured while engine is working.
 * 
 * - Enum type for errors' codes. One of this code is always in last error 
 * variable (variable 'last error' is invisible for programmers - they should 
 * use function engine_error() and optionally engine_error_message() to get know
 * what kind of error occured).
 */
typedef enum
{
        /**
         * :: there was no error - last action successed
         */
        ENGINE_NO_ERROR = 0,
        /**
         * :: file, which engine tried to read, has wrong format or 
         * it is corrupted.
         */
        ENGINE_WRONG_FILE,
        /**
         * :: user do not have permission to read/open file, which engine tried
         * to open/read.
         */
        ENGINE_COULDNT_READ,
        /**
         * :: file, on which engine tried to operate, do not exist
         */
        ENGINE_NO_FILE,     
        /**
         * :: while engine was working, "Out Of Memory" error occured
         */
        ENGINE_OUT_OF_MEMORY
}
EngineStatus;
//------------------------------------------------------------------------------


//______________________________________________________________________________
// *****************************************************************************
//********************************************* DEFINITIONS OF CALLBACK'S TYPES:
//------------------------------------------------------------------------------
/** \brief Type of callback functions for retrieving word list.
 * 
 * Function which is set by dict_eng_set_callback() to signal 
 * ENGINE_WORD_LIST_SIGNAL should be exactly this type.
 *
 * @param list :: <b>GArray*</b> - there will be all found words in this array
 * @param pattern :: <b>gchar*</b> - pattern to which we have been looking for 
 *      word lists
 * @param error :: <b>EngineStatus</b> - error code; if everything was ok it is 
 *      ENGINE_NO_ERROR
 * @param user_data :: <b>gpointer</b> - data set by user to be passing to each 
 *      callback for ENGINE_WORD_LIST_SIGNAL signal
 */
typedef void (*cb_word_list)(GArray* list,
                             gchar* pattern,
                             gpointer user_data,
                             EngineStatus error);
//------------------------------------------------------------------------------
/** \brief Type of callback functions for retrieving word's translation.
 * 
 * Function which is set by dict_eng_set_callback() to signal 
 * ENGINE_WORD_TRANSLATION_SIGNAL should be exactly this type.
 *
 * @param translation :: <b>gchar*</b> - translation of word <i>word</i>
 * @param word :: <b>gchar*</b> - word which translation we already retrieved
 * @param error :: <b>EngineStatus</b> - error code; if everything was ok it is 
 *      ENGINE_NO_ERROR
 * @param user_data :: <b>gpointer</b> - data set by user to be passing to each 
 *      callback for ENGINE_WORD_TRANSLATION_SIGNAL signal
 */
typedef void (*cb_word_translation)(gchar* translation,
                                    gchar* word,
                                    gpointer user_data,
                                    EngineStatus error);
//------------------------------------------------------------------------------
/** \brief Type of callback functions for updating progress bar.
 * 
 * Function which is set by dict_eng_set_callback() to signal 
 * ENGINE_PROGRESS_OPTIMIZING_SIGNAL should be exactly this type.
 *
 * @param value :: <b>double</b> - current progres of process [0.0 - 1.0]
 * @param user_data :: <b>gpointer</b> - data set by user to be passing to each 
 *      callback for ENGINE_PROGRESS_OPTIMIZING_SIGNAL signal
 * @param error :: <b>EngineStatus</b> - error code; if everything is ok it has 
 *      ENGINE_NO_ERROR value
 */
typedef void (*cb_progress)(gdouble value,
                            gpointer user_data,
                            EngineStatus error);
//------------------------------------------------------------------------------

           
//______________________________________________________________________________
// *****************************************************************************
//**************************************** DEFINITIONS OF BASIC FUNCTIONS TYPES:
//------------------------------------------------------------------------------
typedef struct _Engine Engine;
/** \brief Type of function getting some chars from engine.
 *
 * Function of this type should be used for functions which intention is to get 
 * some string from concrete dictionary.
 *
 * @param engine :: <i>Engine*</i> - pointer to engine on which this functions 
 *      has to work
 * @return <i><b>gchar*</b></i> :: chars which represents some information 
 *      depends on concrete functions (e.g engine path etc.)
 */
typedef gchar*   (*getting_char)(Engine* engine);
//------------------------------------------------------------------------------
/** \brief Type of function getting boolean statement from engine.
 *
 * Function of this type should be used for functions which intention is to get 
 * boolean statement from concrete dictionary (e.g. to get know if engine has 
 * cache file - is optimized). 
 *
 * @param engine :: <i>Engine*</i> - pointer to engine on which this functions
 *      has to work
 * @return <i><b>gboolean</b></i> :: gboolean statement which gives programmers 
 *      some information about state of dictionary/engine
 */
typedef gboolean (*getting_bool)(Engine* engine);
//------------------------------------------------------------------------------
/** \brief Type of function which hasn't got to return any value.
 *
 * Function of this type should be used for functions which intention is only 
 * to do something on concrete dictionary (e.g. to make cache file etc).
 *
 * @param engine :: <i>Engine*</i> - pointer to engine on which this functions 
 *      has to work
 * @return nothing. Functions of this type should only perform some action - 
 *      there is no result
 */
typedef void     (*doing_only)(Engine* engine);
//------------------------------------------------------------------------------
/** \brief Type of function which can find words list from dictionary.
 * 
 * Function of this type should be used as a function for searching words list 
 * matching to pattern.
 *
 * @param engine :: <i>Engine*</i> - pointer to engine on which this functions 
 *      has to work
 * @param pattern :: <i>gchar*</i> - pointer to chars which describes pattern 
 *      for searching words
 * @return nothing. This function do not return anything - result should be 
 *      retrieve by callbacks 
 */
typedef void     (*searching_word_list)(Engine* engine, gchar* pattern); 
//------------------------------------------------------------------------------
/** \brief Type of function which can find word's translation from dictionary.
 * 
 * Functions of this type should be used as a function for searching word 
 * translation in concrete dictionary.
 *
 * @param engine :: <i>Engine*</i> - pointer to engine on which this functions 
 *      has to work
 * @param word :: <i>gchar*</i> - pointer to chars which tells what word's 
 *      translation we are looking for
 * @return nothing. This function do not return anything - result should be 
 *      retrieve by callback
 */
typedef void     (*searching_word_translation)(Engine* engine, gchar* word);
//------------------------------------------------------------------------------
/** \brief Type of function which can trans. error code to descriptive message.
 * 
 * Functions of this type should be used as <i>translator</i> from error code 
 * to meaningful message.
 *
 * @param error :: <i>EngineStatus</i> - error to which we want get description
 * @return <i><b>gchar*</b></i> :: chars which describes us what kind of error 
 * occured (e.g. "Dictionary file has wrong file format or it is corrupted.")
 */
typedef gchar*   (*error_to_message)(EngineStatus error);
//------------------------------------------------------------------------------
/** \brief Type of function which should be used to set callbacks for signals.
 * 
 * Functions of this type should be used as a <i>setter</i> for new callback 
 * functions
 *
 * @param engine :: <i>Engine*</i> - pointer to engine on which this functions 
 *      has to work
 * @param signal  :: <i>gchars*</i> - pointer to chars which tell us what kind 
 *      of event should new_functions handle (e.g. "word_list_found", 
 *      "translation_found" etc.)
 * @param c_handler :: <i>gpointer</i> - new callback function
 * @param user_data :: <i>gpointer</i> - pointer do structure, which should be 
 *      passed as a last argument when calling new mounted callback
 * @return <i><b>gpointer</b></i> :: gpointer to previous callback. If NULL 
 *      there could be some error so programmers should check by engine_error() 
 *      function if there was some (if NULL and engine_error() return 
 *      ENGINE_NO_ERROR that means that this callback was unused before).
 * \sa ENGINE_WORD_LIST_SIGNAL :|: ENGINE_WORD_TRANSLATION_SIGNAL :|: 
 * dict_eng_set_callback() :|: cb_word_list :|: cb_word_translation
 */
typedef gpointer (*setting_callbacks)(Engine* engine, 
                                      gchar* signal, 
                                      gpointer c_handler, 
                                      gpointer user_data);
//------------------------------------------------------------------------------
/** \brief Type of function which can get some int value from dictionary.
 * 
 * Functions of this type should be used if programmer want to get some int 
 * value from dictionary.
 *
 * @param enigne :: <i>Engine*</i> - pointer to engine on which this functions 
 *      has to work
 * @return <i><b>gint</b></i> :: int value which user wanted to get from 
 *      dictionary
 */
typedef gint (*getting_int)(Engine* engine);
//------------------------------------------------------------------------------
typedef EngineStatus (*getting_status)(Engine* engine);
//------------------------------------------------------------------------------
typedef void (*setting_progress_seed)(Engine* engine,
                                      gchar* signal,
                                      gdouble seed);
//------------------------------------------------------------------------------


//______________________________________________________________________________
// *****************************************************************************
//*********************************** DEFINITIONS OF ADDITIONAL FUNCTIONS TYPES:
//------------------------------------------------------------------------------
/** \brief Additional functions type.
 * 
 * Functions of this type should be used to check if dictionary in location 
 * could be handled by this engine.
 *
 * @param location :: <i>gchar*</i> - chars which describe location of 
 *      dictionary to check
 * @return <i><b>gboolean</i></b> :: statement telling us if this module could
 *      or could not handle dictionary under location.
 */
typedef gboolean (*checking_compatibiliti)(gchar* location);
//------------------------------------------------------------------------------
/** \brief Additional functions type.
 * 
 * Functions of this type should be used to get some descriptions from 
 * engine/module. Information retrievs by this kind of functions are global for
 * module (e.g: description of handles fromat, description of module etc.).
 *
 * @return <i><b>gchar*</i></b> :: chars which we wanted
 */
typedef gchar*   (*getting_chars_engine)(void);
//------------------------------------------------------------------------------
/** \brief Additional functions type.
 * 
 * Functions of this type should be used to opening dictionary. 
 * @param location :: <i>gchar*</i> - chars which describe where is the 
 *      dictionary which we want to open
 * @param flags :: <i>EngineOptimizationFlag</i> - flags to initialize/use 
 *      optimization methods for opening dictionary.
 * @return <i><b>gpointer</i></b> ::pointer to opened Engine. If NULL there was
 *      some error - programmers should check error type by calling 
 *      engine_error() function.
 */
typedef Engine*  (*creating_engine)(gchar* location,
                                    EngineOptimizationFlag flags,
                                    cb_progress progress_handler,
                                    gpointer progress_data,
                                    gdouble seed);
//------------------------------------------------------------------------------
/** \brief General dictionary structure.
 *  
 * This structure contains every data which are needed to make dictionary 
 * working. Data of dictionary are invisible for user of finnal module. User can
 * operate on dictionary with functions to which there are pointers in this 
 * structure also. engine_data is for module programmer use only. Programmer of 
 * module can put there any data he want. He should not give end user any 
 * information about engine_data structure.
 */
struct _Engine
{
        void (*engine_set_auto_free)(Engine* engine, gboolean state);


        /** \brief Getting location of dictionary.
        *
        * User can use this function if want to get location of concrete 
        * dictionary.
        * \sa getting_char
        */
        getting_char           engine_location;

        /** \brief Getting information if this dictionary is cached.
        *
        * User can use this function if want to know is dictionary cached 
        * actually.
        * \sa getting_bool
        */
        getting_bool           engine_is_optimized;

        /** \brief Making cache file for dictionary.
        *
        * User can use this function if want to make cache file for concrete 
        * dictionary.
        * \sa doing_only
        */
        doing_only                engine_optimize;

        /** \brief Searching word list.
        *
        * User call this function to start searching for word list.
        * \sa searching_word_list
        */
        searching_word_list        engine_search_word_list;

        /** \brief Searching word's translation.
        *
        * User call this function to start searching for word's translation.
        * \sa searching_word_translation
        */
        searching_word_translation   engine_search_word_translation;

        /** \brief Searching word's translation (preformatted HTML).
        *
        * User call this function to start searching for word's translation. 
        * It differs from engine_search_word_translation that it returns 
        * preformatted string (HTML format).
        * \sa searching_word_translation
        */
        searching_word_translation   engine_search_word_translation_extended;        

        /** \brief Close dictionary.
        *
        * User call this if work with dictionary is finished and there is no 
        * need to keep this further working.
        * \sa doing_only
        */
        doing_only                engine_close;

        /** \brief Functions return laste error which occured in dictionary.
        *
        * Programmer should use this function after each action to check if 
        * there was not any errors.
        * \sa getting_int
        */
        getting_status                  engine_error;

        /** \brief <i>Tranlsate</i> error code to meaningful message.
        *
        * Programmer can call this function if want to get meaningful message 
        * describes concrete error code
        * \sa error_to_message
        */
        error_to_message            engine_error_message;

        /** \brief Function to setting any callback for this dictionary.
        * 
        * This function can be used to set some callback function for concrete 
        * dictionary. Programmer should always set callback "word_list_found" 
        * and "word_translation_found". Developers of new modules should also 
        * define for what kind of event end user will be able to set callback.
        */
        setting_callbacks     engine_set_callback;


        setting_progress_seed   engine_set_progress_seed;

        /** \brief Module's private data area. Used only by module programmer.
        * 
        * Here module programmer should keep their own data (e.g. some data for 
        * searching optimalization etc.). Programmer whose only using module, 
        * should not use engine_data at all.
        */
        gpointer engine_data;
};
//------------------------------------------------------------------------------
/** \brief Structure that has pointers to helpfull functions for programmer.
 *
 * There are pointers to some functions in this structure. Programmer whom using 
 * module could use it to get some useful informationa about module (e.g: format
 * handled by module, it's version etc.). The most important functions are: \n
 * engine_check() - to check if this module can handle concrete location \n
 * engine_open() - to open location
 */
typedef struct  {
        
        /** \brief Function to check compatibility beetwen module and location.
        *    
        * Helps programmer to check if this concrete module was designed to work
        * with dictionary under location passed as a argument to this function. 
        * \sa checking_compatibiliti
        */
        checking_compatibiliti          engine_check;      

        /** \brief Function to get some short description of module/engine.
         *      
         * This function returns short description of module, to make possible 
         * displaying for end user what kind of module he is using.
         * \sa getting_chars_engine
         */  
        getting_chars_engine       engine_description;

        /** \brief Function to get module's format.
        *    
        * This function returns format of dictionaries which this module handles
        * \sa getting_chars_engine
        */
        getting_chars_engine       engine_format;

        /** \brief Function to get version of module.
        *    
        *  This functions return version of this module/engine.
        * \sa getting_chars_engine
        */
        getting_chars_engine       engine_version;

        /** \brief Function to open/create new dictionary.
        *    
        * This functions allow programmer to open/create module from concrete 
        * location.
        * \sa creating_engine
        */
        creating_engine                 engine_create;
} 
EngineModule;
//------------------------------------------------------------------------------
/** \brief Additional functions type.
 * 
 * There should be in each module exactly one function of this type named: 
 * engine_global_functions. This function makes easier to load dynamically 
 * module. For doing this then you need only to mapped only this one function
 * and the rest will be avaiable from returned structure EngineModule.
 *
 * @return  <i><b>EngineModule</b></i> :: structure EngineModule from which you 
 *      can call next functions
 * \sa engine_global_functions()
 */
typedef EngineModule (*getting_additional)(void);
//------------------------------------------------------------------------------
/** \brief Function that helps programmer to load basic functions from module. 
 *
 * This function helps programmers to dynamically load module to their 
 * programms. Only this function needed to be mapped from *.so file. Rest of 
 * function are in returned structure EngineModule. User of module could mapped 
 * himslef others functions from concrete modules, but it is not recommended as 
 * far as developers of module could, in fact, give different names for 
 * functions. In structure EngineModule names will be always the same. \n
 * NOTE for module developer: this function should be always defined in your
 * code, to help next programmers whose will be using yours module in theirs own
 * programms.
 * 
 * @return <i><b>EngineModule</b></i> :: Structure EngineModule which members 
 *      are functions for basic operations for this module
 */
extern EngineModule engine_global_functions(void);
//------------------------------------------------------------------------------
#define dict_eng_get_global_functions(library, variable) \
	g_module_symbol ( (library), \
 			  _GLOBAL_FUNCTIONS_NAME_, \
 			  (gpointer)&(variable) \
		        )
          
#ifdef __cplusplus
}                     
#endif     // END OF:    extern "C" {

#endif     // END OF:    _DICTIONARY_ENGINE_BASE
